<?php
// includes/db.php
declare(strict_types=1);

if (session_status() !== PHP_SESSION_ACTIVE) session_start();

// Escape HTML (para vistas)
function h($str): string {
    return htmlspecialchars((string)$str, ENT_QUOTES, 'UTF-8');
}

function app_config(): array {
  static $cfg = null;
  if ($cfg !== null) return $cfg;

  $path = __DIR__ . '/config.php';
  if (!is_file($path)) {
    // Fallback ultra básico si no existe config.php (no recomendado)
    $cfg = [
      'db' => [
        'host' => 'localhost',
        'name' => 'DB_NAME',
        'user' => 'DB_USER',
        'pass' => 'CAMBIAR_PASSWORD',
        'charset' => 'utf8mb4',
      ],
      'store_phone' => '',
    ];
    return $cfg;
  }

  $cfg = require $path;
  if (!is_array($cfg)) $cfg = [];
  return $cfg;
}

/**
 * PDO singleton
 */
function pdo(): PDO {
  static $pdo = null;
  if ($pdo instanceof PDO) return $pdo;

  $cfg = app_config();
  $db  = $cfg['db'] ?? [];

  if (!extension_loaded('pdo_mysql')) {
    throw new RuntimeException('Extensión pdo_mysql no está habilitada en este servidor.');
  }

  $host    = (string)($db['host'] ?? 'localhost');
  $port    = (string)($db['port'] ?? '');
  $socket  = (string)($db['socket'] ?? '');
  $name    = (string)($db['name'] ?? '');
  $user    = (string)($db['user'] ?? '');
  $pass    = (string)($db['pass'] ?? '');
  $charset = (string)($db['charset'] ?? 'utf8mb4');

  if ($name === '' || $user === '') {
    throw new RuntimeException('Config DB incompleta (name/user). Revisa includes/config.php o includes/config.local.php.');
  }

  // DSN builder (socket tiene prioridad)
  $mkDsn = function(string $h) use ($socket, $port, $name, $charset): string {
    if ($socket !== '') {
      return "mysql:unix_socket={$socket};dbname={$name};charset={$charset}";
    }
    $dsn = "mysql:host={$h};dbname={$name};charset={$charset}";
    if ($port !== '' && ctype_digit($port)) {
      $dsn = "mysql:host={$h};port={$port};dbname={$name};charset={$charset}";
    }
    return $dsn;
  };

  $opt = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES   => false,
  ];

  // Algunos hostings fallan con "localhost" (socket) o con TCP; intentamos fallback.
  $hosts = [$host];
  if (strtolower($host) === 'localhost') $hosts[] = '127.0.0.1';
  if ($host === '127.0.0.1') $hosts[] = 'localhost';

  $last = null;
  foreach (array_unique($hosts) as $h) {
    try {
      $pdo = new PDO($mkDsn($h), $user, $pass, $opt);
      return $pdo;
    } catch (Throwable $e) {
      $last = $e;
    }
  }

  // Si ninguno conectó, re-lanzamos
  throw $last ?: new RuntimeException('No se pudo conectar a la base de datos.');
}
