<?php
// orden_entrega_pdf.php
// Genera un PDF sencillo con los datos del pedido para entrega (SIN costos)

// Importante: NO mostrar errores en pantalla porque rompen el PDF
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/lib/fpdf.php';  // Asegúrate de tener FPDF aquí

// Helper para convertir UTF-8 -> ISO-8859-1 (FPDF trabaja en Latin1)
function pdf_text($s) {
    if ($s === null) return '';
    return iconv('UTF-8', 'ISO-8859-1//TRANSLIT', (string)$s);
}

// ===== Seguridad: solo ciertos roles pueden generar orden de entrega =====
$u = current_user();
if (!$u || !in_array($u['role'], ['ventas','admin','tienda','repartidor'], true)) {
    http_response_code(403);
    echo 'No autorizado.';
    exit;
}

$order_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$order_id) {
    echo 'Falta ID de pedido.';
    exit;
}

$pdo = pdo();

// ===== Pedido + datos del cliente =====
$stmt = $pdo->prepare("
    SELECT 
      o.id,
      o.delivery_date,
      o.delivery_time,
      o.delivery_instructions,
      o.comments,
      o.status,
      o.sales_order,
      u.name  AS cliente,
      u.phone AS phone,
      u.address
    FROM orders o
    JOIN users u ON u.id = o.customer_id
    WHERE o.id = ?
");
$stmt->execute([$order_id]);
$pedido = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$pedido) {
    echo 'Pedido no encontrado.';
    exit;
}

// ===== Items: cantidad y descripción del producto (SIN precios) =====
$stmt = $pdo->prepare("
    SELECT 
      oi.quantity,
      p.name AS producto
    FROM order_items oi
    JOIN products p ON p.id = oi.product_id
    WHERE oi.order_id = ?
");
$stmt->execute([$order_id]);
$items = $stmt->fetchAll(PDO::FETCH_ASSOC);

// ======== PDF ========
$pdf = new FPDF('P', 'mm', 'Letter');
$pdf->AddPage();
$pdf->SetFont('Arial','B',14);

// Título
$pdf->Cell(0, 8, pdf_text('Orden de entrega'), 0, 1, 'C');
$pdf->Ln(2);

$pdf->SetFont('Arial','',11);
$pdf->Cell(0, 6, pdf_text('Pedido #'.$pedido['id']), 0, 1);
$pdf->Cell(0, 6, pdf_text('Orden de compra (OC): '.($pedido['sales_order'] ?: '-')), 0, 1);
$pdf->Cell(0, 6, pdf_text('Cliente: '.$pedido['cliente'].' ('.$pedido['phone'].')'), 0, 1);
$pdf->Cell(0, 6, pdf_text('Dirección: '.$pedido['address']), 0, 1);
$pdf->Cell(0, 6, pdf_text('Fecha de entrega: '.$pedido['delivery_date']), 0, 1);
if (!empty($pedido['delivery_time'])) {
    $pdf->Cell(0, 6, pdf_text('Hora de entrega: '.$pedido['delivery_time']), 0, 1);
}
$pdf->Ln(4);

// Instrucciones de entrega
if (!empty($pedido['delivery_instructions'])) {
    $pdf->SetFont('Arial','B',11);
    $pdf->Cell(0, 6, pdf_text('Instrucciones de entrega:'), 0, 1);
    $pdf->SetFont('Arial','',11);
    $pdf->MultiCell(0, 6, pdf_text($pedido['delivery_instructions']));
    $pdf->Ln(2);
}

// Comentarios del cliente
if (!empty($pedido['comments'])) {
    $pdf->SetFont('Arial','B',11);
    $pdf->Cell(0, 6, pdf_text('Comentarios del cliente:'), 0, 1);
    $pdf->SetFont('Arial','',11);
    $pdf->MultiCell(0, 6, pdf_text($pedido['comments']));
    $pdf->Ln(2);
}

// Tabla de productos (SOLO cantidad + descripción)
$pdf->SetFont('Arial','B',11);
$pdf->Cell(30, 7, pdf_text('Cantidad'), 1, 0, 'C');
$pdf->Cell(0, 7, pdf_text('Producto'), 1, 1, 'C');

$pdf->SetFont('Arial','',11);
if (!$items) {
    $pdf->Cell(30, 7, pdf_text('-'), 1, 0, 'C');
    $pdf->Cell(0, 7, pdf_text('Sin items registrados'), 1, 1);
} else {
    foreach ($items as $it) {
        $pdf->Cell(30, 7, pdf_text($it['quantity']), 1, 0, 'C');
        $pdf->Cell(0, 7, pdf_text($it['producto']), 1, 1);
    }
}

$pdf->Ln(10);
$pdf->Cell(0, 6, pdf_text('No,bre y Firma de quien recibe: ___________________________'), 0, 1);
$pdf->Ln(8);
$pdf->Cell(0, 6, pdf_text('Nombre y Firma de repartidor: _____________________________ Fecha de entrega: ___/___/_____'), 0, 1);

// Nombre de archivo y salida (descarga directa)
$filename = 'orden_entrega_'.$pedido['id'].'.pdf';
$pdf->Output('D', $filename);
exit;
