<?php
// whatsapp_helper.php
// Envío WhatsApp con Twilio (Producción o Sandbox) - PHP 8.1+
// Objetivo: NO tronar en VPS donde php_uname() puede estar deshabilitada.

declare(strict_types=1);

/**
 * SHIM defensivo:
 * Algunos vendors viejos de Twilio han intentado llamar funciones en namespace Twilio\Base.
 * Además, en servidores hardened, php_uname() puede estar deshabilitada y causar:
 * "Call to undefined function php_uname()".
 *
 * Aquí definimos wrappers que NO llaman php_uname si no existe.
 */
namespace Twilio {
    if (!function_exists(__NAMESPACE__ . '\\Basephp_uname')) {
        function Basephp_uname(string $mode = 'a') {
            // Si php_uname está deshabilitada, function_exists suele devolver false
            return \function_exists('php_uname') ? \php_uname($mode) : 'unknown';
        }
    }
}
namespace Twilio\Base {
    if (!function_exists(__NAMESPACE__ . '\\php_uname')) {
        function php_uname(string $mode = 'a') {
            return \function_exists('php_uname') ? \php_uname($mode) : 'unknown';
        }
    }
}

namespace {
    // ---------------------------
    // Carga SDK (si existe)
    // ---------------------------
    $WA_HAS_SDK = false;

    $cfgAll = function_exists('app_config') ? (app_config() ?: []) : [];
    $vendorFromCfg = (string)($cfgAll['vendor_autoload'] ?? '');
    $vendorFromEnv = (string)(getenv('TWILIO_VENDOR_AUTOLOAD') ?: '');

    $candidates = array_values(array_filter(array_map('trim', [
        $vendorFromEnv,
        $vendorFromCfg,
        __DIR__ . '/vendor/autoload.php',
    ])));

    foreach ($candidates as $autoload) {
        if ($autoload !== '' && is_file($autoload)) {
            require_once $autoload;
            $WA_HAS_SDK = true;
            break;
        }
    }

    function wa_available(): bool {
        return class_exists(\Twilio\Rest\Client::class);
    }

    /**
     * Normaliza "whatsapp:+E164"
     */
    function wa_norm_from(string $from): string {
        $from = trim($from);
        if ($from === '') return '';
        if (stripos($from, 'whatsapp:') === 0) return $from;

        $norm = preg_replace("/[^0-9\+]/", "", $from) ?? $from;
        $norm = trim($norm);
        if ($norm !== '' && $norm[0] !== '+') $norm = '+' . $norm;
        return 'whatsapp:' . $norm;
    }

    /**
     * Configuración Twilio WhatsApp
     *
     * Lee de:
     *  - app_config()['twilio'] (includes/config.local.php)
     *  - variables de entorno
     */
    function wa_config(): array {
        $cfgAll = function_exists('app_config') ? (app_config() ?: []) : [];
        $tw     = is_array($cfgAll['twilio'] ?? null) ? $cfgAll['twilio'] : [];

        $sidEnv   = (string)(getenv('TWILIO_ACCOUNT_SID') ?: '');
        $tokenEnv = (string)(getenv('TWILIO_AUTH_TOKEN') ?: '');

        $fromEnv  = (string)(getenv('TWILIO_WHATSAPP_FROM') ?: '');
        $msEnv    = (string)(getenv('TWILIO_WHATSAPP_MESSAGING_SERVICE_SID') ?: '');
        $contentEnv = (string)(getenv('TWILIO_WHATSAPP_CONTENT_SID') ?: '');

        $defaultCountryEnv = (string)(getenv('TWILIO_DEFAULT_COUNTRY') ?: 'MX');

        $sid   = trim((string)($tw['sid']   ?? $sidEnv));
        $token = trim((string)($tw['token'] ?? $tokenEnv));

        // FROM oficial WhatsApp (si usas Messaging Service SID, puede ir vacío)
        $fromRaw = trim((string)($tw['wa_from'] ?? $fromEnv));
        $from    = $fromRaw !== '' ? wa_norm_from($fromRaw) : '';

        $msSid     = trim((string)($tw['wa_messaging_service_sid'] ?? $msEnv));
        $contentSid = trim((string)($tw['wa_content_sid'] ?? $contentEnv));
        $defaultCountry = strtoupper(trim((string)($tw['default_country_iso'] ?? $defaultCountryEnv)));
        if ($defaultCountry === '') $defaultCountry = 'MX';

        return [
            'sid'   => $sid,
            'token' => $token,
            'from'  => $from,
            'messaging_service_sid' => $msSid,
            'content_sid' => $contentSid,
            'default_country_iso' => $defaultCountry,
        ];
    }

    /**
     * Normaliza destino a whatsapp:+E164
     * Nota MX: si llega +52 + 10 dígitos, WA suele requerir +521 + 10.
     */
    function wa_phone_to_whatsapp(string $raw, string $country = 'MX'): string {
        $raw = trim($raw);
        if ($raw === '') return '';

        // Quita prefijo whatsapp:
        $raw = preg_replace('/^whatsapp:\s*/i', '', $raw) ?? $raw;
        $raw = trim($raw);

        $country = strtoupper(trim($country ?: 'MX'));

        // Mantener + si viene
        $digitsPlus = preg_replace('/[^0-9\+]/', '', $raw) ?? '';
        $digitsPlus = trim($digitsPlus);

        if ($digitsPlus !== '' && $digitsPlus[0] === '+') {
            $only = preg_replace('/\D/', '', $digitsPlus) ?? '';

            // MX: +52 + 10 => whatsapp:+521 + 10
            if (strpos($only, '52') === 0 && strlen($only) === 12) {
                $last10 = substr($only, -10);
                return 'whatsapp:+521' . $last10;
            }
            // MX ya con 521
            if (strpos($only, '521') === 0 && strlen($only) === 13) {
                return 'whatsapp:+' . $only;
            }
            // US típico
            if (strpos($only, '1') === 0 && strlen($only) === 11) {
                return 'whatsapp:+' . $only;
            }
            // Otro E.164
            return 'whatsapp:' . $digitsPlus;
        }

        // Sin + -> solo dígitos
        $only = preg_replace('/\D/', '', $digitsPlus) ?? '';
        if ($only === '') return '';

        if ($country === 'US') {
            if (strlen($only) === 11 && $only[0] === '1') $only = substr($only, 1);
            if (strlen($only) !== 10) return '';
            return 'whatsapp:+1' . $only;
        }

        // Default MX
        if (strlen($only) < 10) return '';
        $last10 = substr($only, -10);
        return 'whatsapp:+521' . $last10;
    }

    function wa_phone_to_wa_me(string $raw): string {
        $raw = trim($raw);
        if ($raw === '') return '';
        $raw = preg_replace('/^whatsapp:/i', '', $raw) ?? $raw;
        return preg_replace('/\D/', '', $raw) ?? '';
    }

    function wa_make_link(string $phoneRawOrE164, string $message): string {
        $digits = wa_phone_to_wa_me($phoneRawOrE164);
        if ($digits === '') return '';
        return "https://wa.me/{$digits}?text=" . rawurlencode($message);
    }

    /**
     * Envío detallado (para debug)
     */
    function wa_send_one_result(string $to, string $message, string $countryHint = 'MX', array $options = []): array {
        if (!wa_available()) {
            return ['ok' => false, 'error' => 'WA_SDK_NOT_AVAILABLE'];
        }

        $cfg = wa_config();
        if ($cfg['sid'] === '' || $cfg['token'] === '') {
            return ['ok' => false, 'error' => 'WA_MISSING_SID_OR_TOKEN'];
        }

        $toNorm = wa_phone_to_whatsapp($to, $countryHint);
        if ($toNorm === '' || stripos($toNorm, 'whatsapp:') !== 0) {
            return ['ok' => false, 'error' => 'WA_INVALID_TO'];
        }

        $params = [];

        // Prefer Messaging Service si existe
        if (!empty($cfg['messaging_service_sid'])) {
            $params['messagingServiceSid'] = $cfg['messaging_service_sid'];
        } else {
            if (empty($cfg['from'])) return ['ok' => false, 'error' => 'WA_MISSING_FROM'];
            $params['from'] = $cfg['from'];
        }

        // Plantillas (Content API) si se requiere
        $useTemplate = (bool)($options['use_template'] ?? false);
        $contentSid  = trim((string)($options['content_sid'] ?? ($cfg['content_sid'] ?? '')));
        $contentVars = $options['content_vars'] ?? null;

        if ($useTemplate) {
            if ($contentSid === '') return ['ok' => false, 'error' => 'WA_TEMPLATE_CONTENT_SID_REQUIRED'];
            $params['contentSid'] = $contentSid;
            if (is_array($contentVars) && $contentVars) {
                $params['contentVariables'] = json_encode($contentVars, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
            }
        } else {
            $params['body'] = $message;
        }

        try {
            $client = new \Twilio\Rest\Client($cfg['sid'], $cfg['token']);
            $msg = $client->messages->create($toNorm, $params);

            return ['ok' => true, 'sid' => (string)($msg->sid ?? '')];
        } catch (\Throwable $e) {
            // Intentar extraer datos útiles si es RestException
            $twCode = null; $twStatus = null;
            if (class_exists(\Twilio\Exceptions\RestException::class) && $e instanceof \Twilio\Exceptions\RestException) {
                $twCode = method_exists($e, 'getCode') ? $e->getCode() : null;
                $twStatus = method_exists($e, 'getStatusCode') ? $e->getStatusCode() : null;
            }

            return [
                'ok' => false,
                'error' => 'WA_SEND_FAILED',
                'twilio_code' => $twCode,
                'http_status' => $twStatus,
                'detail' => $e->getMessage(),
            ];
        }
    }

    function wa_send_one(string $to, string $message, string $countryHint = 'MX'): bool {
        $r = wa_send_one_result($to, $message, $countryHint, ['use_template' => false]);
        return (bool)($r['ok'] ?? false);
    }
}
